package gov.va.med.domain.service.messaging.transceiver;

import gov.va.med.domain.service.messaging.MessagingLogHelper;
//import gov.va.med.domain.model.Function;
import gov.va.med.domain.service.messaging.Request;
import gov.va.med.domain.service.messaging.decode.MessageDecoder;
import gov.va.med.domain.service.messaging.encode.MessageEncoder;
import gov.va.med.domain.service.messaging.environment.Destination;
import gov.va.med.domain.service.messaging.environment.EndPoint;
import gov.va.med.framework.InvalidConfigurationException;
//import gov.va.med.framework.util.ObjectFactory;
import gov.va.med.mhv.rxrefill.data.model.Function;
import gov.va.med.mhv.rxrefill.exception.MHVRuntimeException;
import gov.va.med.mhv.rxrefill.service.impl.PrescriptionServiceImpl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

/**
 * Responsible for constructing various types of transceivers and transceiver facades.
 * <P>
 * As new requirements arise for constructing new Transceiver or Transceiver Facade classes
 * or existing classes that demand new paramater lists,  this class should be updated
 * with supprting static contructXxx methods.
 * <P>
 * Most low level Transceivers only require simple Bean Factory construction however, which
 * is supported by getTransceiver.
 * <P>
 * Requires bean definitions in messagingContext.xml for
 * <UL>
 * <LI> the Encoder used
 * <LI> the Decoder used
 * <LI> the Transceiver used
 * </UL>
 * <P>
 *
 * @author Joel Goldberg
 * @version $Id: TransceiverFactory.java,v 1.10 2005/06/19 17:16:20 joel.goldberg Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
@Component
public class TransceiverFactory {
	
	private static Logger log = LogManager.getLogger(TransceiverFactory.class);

	@Autowired
	private ApplicationContext appContext;
	
	@Autowired
	private MessagingLogHelper messagingLogHelper;
	
	/**
	 * Creates a TransceiverFacade that mediates invocation of
	 * a lower level tranceiever by adding encode and decode behavior.
	 * <P>
	 * @param request - minimally containing a functionType, IPayload and MessageType
	 * @param destination a Destination instance initializied with the configuration
	 * values from the messaging control tables such as FUNCTIONS, ENDPOINTS,
	 * DESTINATION_ENDPOINTS and FUNCTION_ENDPOINTS. 
	 * @return the encode/decode decorator
	 * @throws InvalidConfigurationException if any configuration values
	 *           prevent the transceiver from being constructed, such as function
	 *           in the request not linked to the Destination via FUNCTION_ENDPOINTS. 
	 */
	public TransceiverFacade constructTransceiverFacade(Request request, Destination destination)
	    throws InvalidConfigurationException {

		EndPoint endPoint = destination.selectEndPoint(request);
		Function function = destination.getFunction(endPoint, request); 
        //MessageEncoder encoder = (request.isApplicationAckRequest())? function.getAaEncoderInstance(): function.getEncoderInstance();
		MessageEncoder encoder = null;
		MessageDecoder messageDecoder = null;
		try {
			encoder = (MessageEncoder) ((request.isApplicationAckRequest()) ? appContext.getBean(function.getAaEncoder()) : appContext.getBean(function.getEncoder()));
		} catch (Exception e) {
			log.error(String.format("unable to load requested bean from application context AaEncoder : %s encoder : %s", 
					function.getAaEncoder(), function.getEncoder()), e);
			throw new MHVRuntimeException(String.format("unable to load requested bean from application context AaEncoder : %s encoder : %s", 
					function.getAaEncoder(), function.getEncoder()), e);
		}
		
		try {
			messageDecoder = (MessageDecoder) appContext.getBean(function.getDecoder());
		} catch(Exception e) {
			log.error(String.format("Unable to load requested messageDecoder %s", function.getDecoder()), e);
			throw new MHVRuntimeException(String.format("Unable to load requested messageDecoder %s", function.getDecoder()), e);
		}
                
		return new TransceiverFacade(getTransceiver(endPoint), encoder, messageDecoder, request, this.messagingLogHelper);
	}

	/**
	 * Gets the transceiver bean of the type specified for the destination/endpoint
	 * The Transceiver is populated with the endpoint which also contains
	 * the associated protocol and encoding parameters.
	 * <P>
	 * The bean name must be defined in domainMessagingContext.xml
	 *
	 * @param endPoint EndPoint
	 * @return Transceiver
	 */
	public Transceiver getTransceiver(EndPoint endPoint) {
		String beanName = endPoint.getTransceiver();
		//Transceiver transceiver = (Transceiver)ObjectFactory.loadAndCreateClass(beanName);
		Transceiver transceiver = null;
		try {
			transceiver = (Transceiver) appContext.getBean(beanName);
			transceiver.setEndPoint(endPoint);
		} catch(Exception e) {
			log.error(String.format("Unable to load transceiver from a bean factory %s", beanName), e);
			throw new MHVRuntimeException(String.format("Unable to load transceiver from a bean factory %s", beanName), e);
		}

		return transceiver;
	}
}
